// Settings page JavaScript

const DEFAULT_SERVER_URL = 'http://192.168.1.250:8080';

// DOM elements
const apiKeyInput = document.getElementById('apiKey');
const serverUrlInput = document.getElementById('serverUrl');
const saveBtn = document.getElementById('saveBtn');
const testBtn = document.getElementById('testBtn');
const saveStatus = document.getElementById('saveStatus');
const testResult = document.getElementById('testResult');

// Load saved settings
async function loadSettings() {
  const settings = await chrome.storage.sync.get(['apiKey', 'serverUrl']);

  if (settings.apiKey) {
    apiKeyInput.value = settings.apiKey;
  }

  serverUrlInput.value = settings.serverUrl || DEFAULT_SERVER_URL;
}

// Save settings
async function saveSettings() {
  const apiKey = apiKeyInput.value.trim();
  const serverUrl = serverUrlInput.value.trim() || DEFAULT_SERVER_URL;

  if (!apiKey) {
    showStatus('error', 'API key is required');
    return;
  }

  await chrome.storage.sync.set({
    apiKey: apiKey,
    serverUrl: serverUrl
  });

  showStatus('success', 'Settings saved successfully!');
}

// Test connection
async function testConnection() {
  const apiKey = apiKeyInput.value.trim();
  const serverUrl = serverUrlInput.value.trim() || DEFAULT_SERVER_URL;

  if (!apiKey) {
    showStatus('error', 'Please enter an API key first');
    return;
  }

  // Show test result area
  testResult.classList.add('visible');

  // Reset values
  setTestValue('testConnection', 'Testing...', '');
  setTestValue('testApiKey', 'Testing...', '');
  setTestValue('testLastSync', '-', '');
  setTestValue('testTotalOrders', '-', '');

  try {
    // Test 1: Connection
    const response = await fetch(`${serverUrl}/api/target/sync-status`, {
      method: 'GET',
      headers: {
        'Accept': 'application/json'
      }
    });

    if (!response.ok) {
      throw new Error(`Server returned ${response.status}`);
    }

    setTestValue('testConnection', 'Connected', 'success');

    const data = await response.json();

    // Test 2: Validate API key by making a test request
    const keyTest = await fetch(`${serverUrl}/api/target/api-key`, {
      method: 'GET',
      headers: {
        'Accept': 'application/json'
      }
    });

    if (keyTest.ok) {
      // API endpoint is accessible, key format will be validated on actual import
      setTestValue('testApiKey', 'Format OK', 'success');
    } else {
      setTestValue('testApiKey', 'Unable to verify', 'error');
    }

    // Show sync status
    if (data.last_sync) {
      const lastSync = new Date(data.last_sync);
      const daysAgo = data.days_since_sync;
      setTestValue('testLastSync', daysAgo === 0 ? 'Today' : `${daysAgo} days ago`, 'success');
    } else {
      setTestValue('testLastSync', 'Never', '');
    }

    setTestValue('testTotalOrders', data.total_orders.toString(), 'success');

    showStatus('success', 'Connection test successful!');

  } catch (error) {
    setTestValue('testConnection', 'Failed', 'error');
    setTestValue('testApiKey', '-', '');
    showStatus('error', `Connection failed: ${error.message}`);
  }
}

// Show status message
function showStatus(type, message) {
  saveStatus.className = 'status ' + type;
  saveStatus.textContent = message;

  // Auto-hide success messages
  if (type === 'success') {
    setTimeout(() => {
      saveStatus.className = 'status';
    }, 3000);
  }
}

// Set test result value
function setTestValue(id, value, status) {
  const el = document.getElementById(id);
  el.textContent = value;
  el.className = 'value' + (status ? ' ' + status : '');
}

// Event listeners
saveBtn.addEventListener('click', saveSettings);
testBtn.addEventListener('click', testConnection);

// Load settings on page load
loadSettings();
