// State management
let currentState = 'loading';

// DOM elements
const states = {
  notConfigured: document.getElementById('notConfigured'),
  notOnTarget: document.getElementById('notOnTarget'),
  readyToSync: document.getElementById('readyToSync'),
  syncing: document.getElementById('syncing'),
  results: document.getElementById('results'),
  error: document.getElementById('error')
};

// Show a specific state
function showState(state) {
  Object.values(states).forEach(el => el.classList.add('hidden'));
  if (states[state]) {
    states[state].classList.remove('hidden');
  }
  currentState = state;
}

// Check configuration
async function checkConfig() {
  const config = await chrome.storage.sync.get(['apiKey', 'serverUrl']);
  return {
    isConfigured: !!config.apiKey,
    apiKey: config.apiKey,
    serverUrl: config.serverUrl || 'http://192.168.1.250:8080'
  };
}

// Check if we're on Target orders page
async function checkTargetPage() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tab || !tab.url) return false;
    return tab.url.includes('target.com/orders') || tab.url.includes('target.com/account/orders');
  } catch (e) {
    return false;
  }
}

// Initialize popup
async function init() {
  const config = await checkConfig();

  if (!config.isConfigured) {
    showState('notConfigured');
    return;
  }

  const onTargetPage = await checkTargetPage();

  if (!onTargetPage) {
    showState('notOnTarget');
    return;
  }

  showState('readyToSync');
}

// Sync orders
async function syncOrders() {
  showState('syncing');
  document.getElementById('syncStatus').textContent = 'Scanning orders...';

  try {
    // Get configuration
    const config = await checkConfig();
    if (!config.isConfigured) {
      throw new Error('API key not configured');
    }

    // Get the current tab
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

    // Execute content script to scrape orders
    document.getElementById('syncStatus').textContent = 'Extracting order data...';

    const results = await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      func: scrapeTargetOrders
    });

    const orders = results[0]?.result || [];

    if (orders.length === 0) {
      throw new Error('No orders found on this page. Make sure you are on the Target orders page and orders are visible.');
    }

    document.getElementById('syncStatus').textContent = `Sending ${orders.length} orders to budget app...`;

    // Send to budget app
    const response = await fetch(`${config.serverUrl}/api/target/import`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${config.apiKey}`
      },
      body: JSON.stringify({
        source: 'target_extension',
        orders: orders
      })
    });

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.error || `Server returned ${response.status}`);
    }

    const data = await response.json();

    // Show results
    document.getElementById('ordersFound').textContent = data.results.total_received;
    document.getElementById('ordersNew').textContent = data.results.new_imported;
    document.getElementById('ordersSkipped').textContent = data.results.duplicates_skipped;
    document.getElementById('resultSummary').textContent =
      data.results.new_imported > 0
        ? `${data.results.new_imported} new orders imported!`
        : 'All orders already synced';

    showState('results');

  } catch (error) {
    console.error('Sync error:', error);
    document.getElementById('errorMessage').textContent = error.message;
    showState('error');
  }
}

// Scrape Target orders (runs in page context)
function scrapeTargetOrders() {
  const orders = [];

  // Find order cards on the page
  // Target's order history page structure varies, so we try multiple selectors
  const orderElements = document.querySelectorAll('[data-test="orderCard"], [data-test="order-card"], .OrderCard, [class*="OrderCard"], [class*="order-card"]');

  // If no order cards found, try alternative structure
  if (orderElements.length === 0) {
    // Try to find orders in a list format
    const orderLinks = document.querySelectorAll('a[href*="/orders/"]');
    console.log('Target Budget Sync: Found', orderLinks.length, 'order links');
  }

  orderElements.forEach((orderEl, index) => {
    try {
      // Extract order ID from link or data attribute
      const orderLink = orderEl.querySelector('a[href*="/orders/"]');
      let orderId = null;

      if (orderLink) {
        const match = orderLink.href.match(/orders\/(\d+)/);
        if (match) orderId = 'TARGET_' + match[1];
      }

      if (!orderId) {
        orderId = 'TARGET_' + Date.now() + '_' + index;
      }

      // Extract order date
      const dateEl = orderEl.querySelector('[data-test="orderDate"], [class*="date"], time');
      let orderDate = null;
      if (dateEl) {
        const dateText = dateEl.textContent || dateEl.getAttribute('datetime');
        const parsed = new Date(dateText);
        if (!isNaN(parsed)) {
          orderDate = parsed.toISOString().split('T')[0];
        }
      }

      // Extract total amount
      const totalEl = orderEl.querySelector('[data-test="orderTotal"], [class*="total"], [class*="Total"]');
      let totalAmount = 0;
      if (totalEl) {
        const priceMatch = totalEl.textContent.match(/\$?([\d,]+\.?\d*)/);
        if (priceMatch) {
          totalAmount = parseFloat(priceMatch[1].replace(',', ''));
        }
      }

      // Extract order type (in-store, online, pickup)
      let orderType = 'online';
      const typeEl = orderEl.querySelector('[class*="fulfillment"], [class*="Fulfillment"], [class*="delivery"]');
      if (typeEl) {
        const typeText = typeEl.textContent.toLowerCase();
        if (typeText.includes('store') || typeText.includes('in-store')) {
          orderType = 'in_store';
        } else if (typeText.includes('pickup')) {
          orderType = 'pickup';
        } else if (typeText.includes('ship') || typeText.includes('delivery')) {
          orderType = 'shipped';
        }
      }

      // Extract items
      const items = [];
      const itemElements = orderEl.querySelectorAll('[data-test="orderItem"], [class*="item"], [class*="Item"], [class*="product"]');

      itemElements.forEach(itemEl => {
        const nameEl = itemEl.querySelector('[class*="name"], [class*="Name"], [class*="title"], [class*="Title"], h3, h4');
        const priceEl = itemEl.querySelector('[class*="price"], [class*="Price"]');
        const qtyEl = itemEl.querySelector('[class*="qty"], [class*="Qty"], [class*="quantity"], [class*="Quantity"]');

        if (nameEl) {
          const name = nameEl.textContent.trim();
          let price = 0;
          let quantity = 1;

          if (priceEl) {
            const priceMatch = priceEl.textContent.match(/\$?([\d,]+\.?\d*)/);
            if (priceMatch) price = parseFloat(priceMatch[1].replace(',', ''));
          }

          if (qtyEl) {
            const qtyMatch = qtyEl.textContent.match(/(\d+)/);
            if (qtyMatch) quantity = parseInt(qtyMatch[1]);
          }

          items.push({
            name: name,
            quantity: quantity,
            unit_price: price,
            total_price: price * quantity,
            target_category: 'Uncategorized' // Will be categorized by budget app
          });
        }
      });

      // Only add order if we have some data
      if (orderId && (totalAmount > 0 || items.length > 0)) {
        orders.push({
          order_id: orderId,
          order_date: orderDate || new Date().toISOString().split('T')[0],
          order_time: null,
          order_type: orderType,
          store_location: null,
          total_amount: totalAmount,
          payment_method_last4: null,
          items: items
        });
      }
    } catch (e) {
      console.error('Error parsing order:', e);
    }
  });

  console.log('Target Budget Sync: Scraped', orders.length, 'orders');
  return orders;
}

// Event listeners
document.getElementById('syncBtn')?.addEventListener('click', syncOrders);
document.getElementById('syncAgainBtn')?.addEventListener('click', syncOrders);
document.getElementById('retryBtn')?.addEventListener('click', syncOrders);

document.getElementById('openSettingsBtn')?.addEventListener('click', () => {
  chrome.runtime.openOptionsPage();
});

document.getElementById('settingsLink')?.addEventListener('click', (e) => {
  e.preventDefault();
  chrome.runtime.openOptionsPage();
});

// Initialize when popup opens
init();
